// --*- C++ -*------x---------------------------------------------------------
//
// Class:           Vector3D
// 
// Base class:      -
//
// Derived classes: - 
//
// Description:     Vector in 3D space.
// 
// -----------------x-------------------x-------------------x-----------------

#ifndef __VECTOR3D_H__
#define __VECTOR3D_H__

// Includes 

extern "C" {
#include <math.h>
}

#include <iostream>
#include <string>

#include <debug.h>
// #include <Limits.h>

using namespace std;

#ifndef PI
#define PI 3.14159265358979323846	/* pi */
#endif

const double RAD2DEG = 180.0/PI;
const double DEG2RAD = PI/180.0;

// maximum plausible component size
const double V3D_PLAUSIBLE = 10000.0;

/** Vector in 3D space.

  @see    @Gamma et al: "Design Patterns", Bridge-Pattern */
class Vector3D {

protected:
  /* COORDINATES */
  double _x;
  double _y;
  double _z;
  
public:

  Vector3D(double __x=0.0, double __y=0.0, double __z=0.0) {
    _x = __x;
    _y = __y;
    _z = __z;
  }

  Vector3D(const Vector3D& orig)  { copy(orig); }

  Vector3D& operator = (const Vector3D& orig) { 
    if (&orig != this) {
      copy(orig);
      }
    return *this; 
  }
  
  virtual ~Vector3D() { }
  
  /* Selectors */
  
  /** What is x-coordinate? */
  inline double x() const { return _x; }
  /** What is y-coordinate? */
  inline double y() const { return _y; }
  /** What is z-coordinate? */
  inline double z() const { return _z; }
  
  /** How long is vector? */
  inline double length() const { 
    return sqrt(lengthSquare());
  }

  inline double lengthSquare() const { 
    return ((_x*_x) + (_y*_y) + (_z*_z)); 
  }
  
  inline bool operator == (const Vector3D& other) const { 
    return ((_x==other._x) && (_y==other._y) && (_z==other._z));
  }

  inline bool operator != (const Vector3D& other) const { 
    return (!(*this == other)); 
  }

  inline bool operator <  (const Vector3D& other) const{ 
    return lengthSquare() <  other.lengthSquare(); 
  }

  inline bool operator <= (const Vector3D& other) const { 
    return lengthSquare() <= other.lengthSquare(); 
  }

  inline bool operator >  (const Vector3D& other) const { 
    return lengthSquare() >  other.lengthSquare(); 
  }

  inline bool operator >= (const Vector3D& other) const { 
    return lengthSquare() >= other.lengthSquare(); 
  }
  
  inline bool operator <  (double value) const { 
    return lengthSquare() <  (value*value); 
  }
  
  inline bool operator <= (double value) const { 
    return lengthSquare() <= (value*value); 
  }
  
  inline bool operator >  (double value) const { 
    return lengthSquare() >  (value*value); 
  }
  
  inline bool operator >= (double value) const { 
    return lengthSquare() >= (value*value); 
  }

  /* Modifiers */

  /** copy method, used by copy constructor and assignment operator */
  inline void copy(const Vector3D& other) {
    _x = other._x;
    _y = other._y;
    _z = other._z;
  }

  /** Set x-coordinate. */
  inline void x(double d) {
    PRECOND(fabs(d) < V3D_PLAUSIBLE, exception);  _x = d; 
  }

  /** Set y-coordinate. */
  inline void y(double d) {
    PRECOND(fabs(d) < V3D_PLAUSIBLE, exception); _y = d; 
  }

  /** Set z-coordinate. */
  inline void z(double d) {
    PRECOND(fabs(d) < V3D_PLAUSIBLE, exception); _z = d; 
  }

  inline void set(double __x, double __y, double __z) { _x = __x; _y = __y; _z = __z; }

  /** normalize vector according to Euclidian norm */
  inline void normalize() { 
    double f = length();
    if (f > 0.0) {  
      f = 1.0 / f;
      _x *= f;
      _y *= f;
      _z *= f;
      // (*this) = (*this) * (1.0 / l); 
    }
    ASSERT(isSimilar(length(), 1.0), exception); 
  }

  inline Vector3D operator + (const Vector3D& other) const { 
    return Vector3D(_x + other._x, _y + other._y, _z + other._z); 
  }
  
  inline Vector3D operator + (double other) const { 
    return Vector3D(_x + other, _y + other, _z + other); }

  inline Vector3D operator - (double other) const { 
    return Vector3D(_x - other, _y - other, _z - other); }

  inline Vector3D operator - (const Vector3D& other) const {
    return Vector3D(_x - other._x, _y - other._y, _z - other._z); 
  }

  inline Vector3D operator * (double other) const { 
    return Vector3D(_x * other, _y * other, _z * other); 
  }

  inline double   operator * (const Vector3D& other) const { 
    return _x*other._x + _y*other._y + _z*other._z;
  }

  inline Vector3D  operator / (double other) const {
    return Vector3D(_x / other, _y / other, _z / other); 
  }

  inline Vector3D& operator += (const Vector3D& other) {
    _x += other._x;
    _y += other._y;
    _z += other._z;
    return (*this);
  }

  inline Vector3D& operator -= (const Vector3D& other) {
    _x -= other._x;
    _y -= other._y;
    _z -= other._z;
    return (*this);
  }

  inline Vector3D& operator *= (double other) {
    _x *= other;
    _y *= other;
    _z *= other;
    return (*this);
  }

  inline Vector3D& operator /= (double other) {
    _x /= other;
    _y /= other;
    _z /= other;
    return (*this);
  }

  inline Vector3D operator - () const { 
    return Vector3D(-_x, -_y, -_z); 
  }

  /* Friends */

  friend Vector3D cross(const Vector3D& v1, const Vector3D& v2);
  // friend Vector3D rotate(const Vector3D& v, const Vector3D& center, 
  // double angle);
  friend ostream& operator << (ostream& os, const Vector3D& p);
  friend istream& operator >> (istream& is, Vector3D& p);

};

/** What is angle between Vector3D a and b? 

    @return angle from 0 to pi */
inline 
double
angle(const Vector3D& a, const Vector3D& b)
{
  PRECOND(a.length() * b.length() != 0.0, 
	  logic_error("Angle of 0-vector requested."));

  double d = (a * b) / (a.length() * b.length());

  // Cope with numeric inaccurracy.
  if (d > 1.0)
    {
      d = 1.0;
    }
  else if (d < -1.0)
    {
      d = -1.0;
    }

  // INVARIANT((-1.0 <= d) && (d <= 1.0), logic_error);

  return acos(d);
}

/** Cross product. */
inline 
Vector3D 
cross(const Vector3D& v1, const Vector3D& v2) 
{ 
  Vector3D result;
  result._x = v1._y * v2._z - v1._z * v2._y;
  result._y = v1._z * v2._x - v1._x * v2._z;
  result._z = v1._x * v2._y - v1._y * v2._x;
  return result;

}

/** Euclidian distance between two points. */
inline 
double
vecDistance(const Vector3D& a, const Vector3D& b)
{
  return (a-b).length();
}

/** Euclidian distance between two points. */
inline 
double
vecDistanceSquare(const Vector3D& a, const Vector3D& b)
{
  return (a-b).lengthSquare();
}

/** out of plane bend: what is the angle between the plane defined
    with v1 and v2 and the vector v3 ? 
    @author Eckart Bindewald
*/
inline
double
outOfPlaneBend(const Vector3D& v1,const Vector3D& v2,const Vector3D& v3)
{
  Vector3D crossV = cross(v1,v2);
  return fabs( (0.5*PI) - angle(crossV,v3) );
}

inline 
ostream& 
operator << (ostream& os, const Vector3D& v)
{
  return os << "(Vector3D " << v.x() << " " << v.y() << " " << v.z() << " ) ";
}

inline 
istream& 
operator >> (istream& is, Vector3D& v)
{
  double x, y, z;
  string tag1, tag2;

  is >> tag1 >> x >> y >> z >> tag2;

  if ((tag1 != "(Vector3D") || (tag2 != ")")) 
    {
      is.clear(ios::badbit);
    }
  else
    {
      v = Vector3D(x, y, z);
    }

  return is;
}

#endif /* __VECTOR3D_H__ */
