
#ifndef __COMPASS_HELP__
#define __COMPASS_HELP__

#include <SequenceAlignment.h>
#include <SimpleSequenceAlignment.h>
#include <CompensationScorer.h>
#include <Vec.h>
#include <RankedSolution5.h>
#include <Stem.h>
#include <ClassificationResult.h>
#include <Histogram.h>

using namespace std;

// minimum used value for standard deviation of single column information
const double INF_SINGLE_DEV_MIN = 1e-5;

/** reads matrix in format that is also used by program "R" */
/*
Vec<Vec<double> >
readPlainMatrix(istream& is);
*/

void
rescaleWithLength(Vec<Vec<double> >& compMatrix, 
		  double lengthScaleWeight, 
		  double lengthScaleMean,
		  unsigned int length);

void
rescaleWithDistance(Vec<Vec<double> >& compMatrix, 
		    double distanceScaleWeight, 
		    double distanceScaleMean);

/** reduces all scores of matrix elements that have some gaps. Multiply with pow(f, alpha) with f being fraction of gaps */
void
gapFracRescale(Vec<Vec<double> >& matrix, const SequenceAlignment& ali, double alpha);

/** sets all matrix elements to zero that correspond to alignment columns with less than numCharFilter non-gap characters */
void
entropyRescale(Vec<Vec<double> >& matrix, const SequenceAlignment& ali, double alpha);

/** filteres out bad rows and columns */
Vec<unsigned int>
filterBadColumns(Vec<Vec<double> >& compMatrix, double threshold, unsigned int maxAbove);

/** sets all matrix elements to zero that correspond to alignment columns with less than numCharFilter non-gap characters */
void
charMinFilter(Vec<Vec<double> >& matrix, const SequenceAlignment& ali, int numCharMinFilter);

/** converts matrix to p-values */
void
convertToPValues(Vec<Vec<double> >& compMatrix, 
		 double pStatAvg, double pStatDev);

void
computeInformationZScore(const SequenceAlignment& ali, 
			 const CompensationScorer& scorer2, 
			 double& informationMean, 
			 double& informationDev,
			 const Vec<double>& wVec);

void
findOptimalSubset(const string& s1, const string& s2,
		  const Vec<double>& aliSequenceWeights,
		  const CompensationScorer& scorer,
		  unsigned int numTrials);

void
findOptimalSubset(const SequenceAlignment& ali,
		  const Vec<unsigned int>& positions,
		  const Vec<double>& aliSequenceWeights,
		  const CompensationScorer& scorer,
		  unsigned int numTrials);

/** generates consensus probability matrix:
    store optional pairing probability matrices of sequences
*/
Vec<Vec<double> >
generateProbabilityMatrix(const Vec<Vec<RankedSolution5<unsigned int, unsigned int> > > probVectors,
			   const Vec<string>& probNames,
			   const SequenceAlignment& ali,
			   char gapChar,
			   const Vec<double>& sequenceWeights);

/** generates consensus probability matrix:
    store optional pairing probability matrices of sequences
    different algorithm: compute consensus instead of average 
*/
Vec<Vec<double> >
generateProbabilityMatrix2(const Vec<Vec<RankedSolution5<unsigned int, unsigned int> > > probVectors,
			   const Vec<string>& probNames,
			   const SequenceAlignment& ali,
			   char gapChar,
			   const Vec<double>& sequenceWeights);

/** generates consensus probability matrix:
    store optional pairing probability matrices of sequences
    different algorithm: compute consensus instead of average 
*/
Vec<Vec<double> >
generateProbabilityMatrix3(const Vec<Vec<RankedSolution5<unsigned int, unsigned int> > > probVectors,
			   const Vec<string>& probNames,
			   const SequenceAlignment& ali,
			   char gapChar,
			   const Vec<double>& sequenceWeights);


/** returns Mathews coefficient between reference stems and prediction matrix*/
double 
computeMatrixMathews(const Vec<Vec<double> >& compMatrix,
		     const Vec<Stem>& referenceStems,
		     double scoreLimit,
		     double& accuracy,
		     double& coverage,
		     double& wrongPosFrac,
		     unsigned int& wrongPos,
		     unsigned int borderLim,
		     unsigned int diagLim);


/** returns Mathews coefficient between reference stems and prediction matrix, 
    computed only for positions which are not a gap character in reference sequence. */
double 
computeMatrixMathews(const Vec<Vec<double> >& compMatrix,
		     const Vec<Stem>& referenceStems,
		     const string& referenceSequence,
		     double scoreLimit,
		     double& accuracy,
		     double& coverage,
		     double& wrongPosFrac,
		     unsigned int& wrongPos,
		     unsigned int borderLim,
		     unsigned int diagLim,
		     char gapChar);

/** returns Mathews coefficient between reference stems and prediction matrix, 
    computed only for positions which are not a gap character in reference sequence.
    accuracy2 is defined according to Juan and Wilson (JMB 1999) */
double 
computeMatrixMathews2(const Vec<Vec<double> >& compMatrix,
		      const Vec<Stem>& referenceStems,
		      const string& referenceSequence,
		      double scoreLimit,
		      double& accuracy,
		      double& accuracy2,
		      double& coverage,
		      double& wrongPosFrac,
		      unsigned int& wrongPos,
		      unsigned int borderLim,
		      unsigned int diagLim,
		      char gapChar);

/** returns Mathews coefficient between reference stems and prediction matrix */
double 
computeMatrixMathews2(const Vec<Vec<double> >& compMatrix,
		      const Vec<Stem>& referenceStems,
		      double scoreLimit,
		      double& accuracy,
		      double& accuracy2,
		      double& coverage,
		      double& wrongPosFrac,
		      unsigned int& wrongPos,
		      unsigned int borderLim,
		      unsigned int diagLim);

/** returns Mathews coefficient between reference stems and prediction matrix,
    accuracy2 is defined according to Juan and Wilson */
void
computeMatrixMathews2(const Vec<Vec<double> >& compMatrix,
		      const Vec<Stem>& referenceStems,
		      double scoreLimit,
		      unsigned int borderLim,
		      unsigned int diagLim,
		      ClassificationResult& classResult);

/** returns Mathews coefficient between reference stems and prediction matrix,
    accuracy2 is defined according to Juan and Wilson */
void
computeMatrixMathews2Intervall(const Vec<Vec<double> >& compMatrix,
			       const Vec<Stem>& referenceStems,
			       double scoreLimitLow,
			       double scoreLimitHigh,
			       unsigned int borderLim,
			       unsigned int diagLim,
			       ClassificationResult& classResult);

/** returns Mathews coefficient between reference stems and prediction matrix,
    accuracy2 is defined according to Juan and Wilson */
void
computeMatrixMathews2IntervallB(const Vec<Vec<double> >& compMatrix,
			       const Vec<Stem>& referenceStems,
			       double scoreLimitLow,
			       double scoreLimitHigh,
			       unsigned int borderLim,
			       unsigned int diagLim,
				   Histogram& basePairHist,
				   Histogram& nobasePairHist,
			       ClassificationResult& classResult);

/** returns Mathews coefficient between reference stems and prediction matrix,
    accuracy2 is defined according to Juan and Wilson */
void
computeMatrixMathews2IntervallStem(const Vec<Vec<double> >& compMatrix,
				   const Vec<Stem>& referenceStems,
				   double scoreLimitLow,
				   double scoreLimitHigh,
				   int stemLen,
				   unsigned int borderLim,
				   unsigned int diagLim,
				   ClassificationResult& classResult);

/** returns Mathews coefficient between reference stems and prediction matrix
 */
void
computeMatrixMathews2IntervallStemSequence(const Vec<Vec<double> >& compMatrix,
					   const string& sequence,
					   const Vec<Stem>& referenceStems,
					   double scoreLimitLow,
					   double scoreLimitHigh,
					   int stemLen,
					   unsigned int borderLim,
					   unsigned int diagLim,
					   ClassificationResult& classResult);

/** writes individualized secondary structures to output file */
Vec<Vec<double> >
writeIndividualStructures(ostream& os, 
			  Vec<Vec<double> > compMatrix, 
			  const CompensationScorer& scorer,
			  const SequenceAlignment& ali, 
			  double individualThreshold,
			  int format,
			  unsigned int stemMinLength);

/** returns index of word. Example: c1==1 c2==2, alphabet={A,C} -> pairs=AA,AC,CA,CC -> result = 1*2+1=3 */
unsigned int
alphabetPairIndex(unsigned int id1, unsigned int id2, unsigned int alphabetSize);

/** returns index of word. Example: c1==C c2==C, alphabet={A,C} -> pairs=AA,AC,CA,CC -> result = 1*2+1=3 */
unsigned int
alphabetPairIndex(char c1, char c2, const string& alphabet);

/** returns all words with length two corresponding to findAlphabetPairIndices */
Vec<string>
generateAlphabetPairs(const string& alphabet);

unsigned int
alphabetPairIndex(char c1, char c2, const string& alphabet);

void
generateMutationMatrices(const SequenceAlignment& ali,
			 const Vec<Vec<double> >& refMatrix,
			 const string& alphabet,
			 Vec<Vec<double> >& contactMatrix,
			 Vec<Vec<double> >& noContactMatrix);

#endif
