// --*- C++ -*------x---------------------------------------------------------
// $Id: Histogram3D.cc,v 1.2 2007/08/14 18:27:51 bindewae Exp $
//
// Class:           Histogram
// 
// Base class:      -
//
// Derived classes: - 
//
// Author:          Eckart Bindewald
//
// Description:     This class implements the concept of a histogram
// 
// -----------------x-------------------x-------------------x-----------------

#include <Histogram3D.h>

// ---------------------------------------------------------------------------
//                                   Histogram
// -----------------x-------------------x-------------------x-----------------

/* CONSTRUCTORS */

/** 
 * default operator
  */
Histogram3D::Histogram3D() : minx(0.0), stepx(1.0),
			     miny(0.0), stepy(1.0), minz(0.0), stepz(1.0)
{
  // no precondition (except sufficient memory)
  // postcondition depends on class
}

/** copy constructor
  */
Histogram3D::Histogram3D(const Histogram3D& orig)
{
  // no precondition (except sufficient memory)
  copy(orig);
  //   POSTCOND( (orig == *this));
}

/** destructor
  */
Histogram3D::~Histogram3D()
{
  // no precondition (except sufficient memory)
  // postcondition depends on class
}


/* OPERATORS */

/** Assignement operator. */
Histogram3D& Histogram3D::operator = (const Histogram3D& orig)
{
  // no precondition
  if (&orig != this)
    {
      copy(orig);
    }
  POSTCOND( (orig == *this));
  return *this;
}

/** comparison operator. 
*/
bool 
operator == (const Histogram3D& left,const Histogram3D& right)
{
  // no precondition
  return left.compare(right);
}

/** output operator 
*/
ostream& 
operator << (ostream& os, const Histogram3D& object)
{
  PRECOND(os);

  os << object.getDimX() << " " << object.minx << " " << object.stepx << " "
     << object.getDimY() << " " << object.miny << " " << object.stepy << " "
     << object.getDimZ() << " " << object.minz << " " << object.stepz << " "
     << object.histData;

  POSTCOND(os);
  return os;
}

/** input operator  
*/
istream& 
operator >> (istream& is, Histogram3D& object)
{
  ERROR_IF(!is, "Error reading histogram data file!");
  int nx, ny, nz; // only dummies
  is >> object.minx >> object.stepx >> nx 
     >> object.miny >> object.stepy >> ny
     >> object.minz >> object.stepz >> nz >> object.histData;

  ERROR_IF(!is, "Error reading histogram!");
  ERROR_IF(object.histData.size() == 0,
	   "No histogram data read!");
  return is;
}


/* PREDICATES */


/**
 * returns true, if all members of other and this object 
 * give true with respect to "==" operator.
 */
bool 
Histogram3D::compare(const Histogram3D& other) const
{
  // no precondition

  ERROR("Compare method not yet implemented!");

  return false;
}

/** returns histogram value for x,y,z */
double 
Histogram3D::getValue(double x, double y, double z) const
{
  ASSERT(histData.size() > 0);
  return histData[getBinX(x)][getBinY(y)][getBinZ(z)];  
}

/** same as getValue, but in case of out of bounds it uses nearest in-bound values */
double 
Histogram3D::getSafeValue(double x, double y, double z) const
{
  ASSERT(histData.size() > 0);
  return histData[getSafeBinX(x)][getSafeBinY(y)][getSafeBinZ(z)];  
}

/*
double 
Histogram3D::getInterpolatedValue(double orig) const
{
  // get bin
  ERROR_IF(histData.size() == 0, "Unspecified histogram data!");
  double result = 0.0;
  // check if out of bounds: 
  if (orig < minx) { 
    result = getBinValue(0); 
  }
  else if (orig > getMax()) {
    result = getBinValue(histData.size() - 1);
  }
  else { // "normal" case
    int b = getBin(orig);
    double middle = getBinMiddle(b);
    
    if (orig >= middle) {
      result = getBinValue(b) + 
	( (orig-middle) * (getBinValue(b+1) - getBinValue(b)) / step );
    }
    else {
      if (b <= 0) {
	return getBinValue(0); // no interpolation of left side of first bin
      }
      double middle2 = middle -step;
      ASSERT(orig >= middle2);
      result = getBinValue(b-1) + 
	(orig-middle2) * (getBinValue(b) - getBinValue(b-1)) / step;
    }
  }
  return result;
}
*/

double 
Histogram3D::getBinValue(index_type nx,
			 index_type ny,
			 index_type nz) const
{
  ERROR_IF(histData.size() == 0, "Unspecified histogram data!");
  if (nx < 0) {
    nx = 0;
  }
  if (nx >= getDimX()) {
    nx = getDimX() - 1;
  }
  if (ny < 0) {
    ny = 0;
  }
  if (ny >= getDimY()) {
    ny = getDimY() - 1;
  }
  if (nz < 0) {
    nz = 0;
  }
  if (nz >= getDimZ()) {
    nz = getDimZ() - 1;
  }
  return histData[nx][ny][nz];
}

int
Histogram3D::getBinX(double d) const
{
  return static_cast<int>((d - minx) / stepx);
}

int
Histogram3D::getBinY(double d) const
{
  return static_cast<int>((d - miny) / stepy);
}

int
Histogram3D::getBinZ(double d) const
{
  return static_cast<int>((d - minz) / stepz);
}

int
Histogram3D::getSafeBinX(double d) const
{
  int bin = getBinX(d);
  if (bin < 0) {
    return 0;
  }
  if (bin >= static_cast<int>(getDimX())) {
    return getDimX()-1;
  }
  return bin;
}

int
Histogram3D::getSafeBinY(double d) const
{
  int bin = getBinY(d);
  if (bin < 0) {
    return 0;
  }
  if (bin >= static_cast<int>(getDimY())) {
    return getDimY()-1;
  }
  return bin;
}

int
Histogram3D::getSafeBinZ(double d) const
{
  int bin = getBinZ(d);
  if (bin < 0) {
    return 0;
  }
  if (bin >= static_cast<int>(getDimZ())) {
    return getDimZ()-1;
  }
  return bin;
}


/* MODIFIERS */

/** copies orig object to this object ("deep copy")
  */
void 
Histogram3D::copy(const Histogram3D& orig)
{
  // no precondition

  histData = orig.histData;
  minx = orig.minx;
  stepx = orig.stepx;
  miny = orig.miny;
  stepy = orig.stepy;
  minz = orig.minz;
  stepz = orig.stepz;

  //   POSTCOND(orig == *this);
}




