// --*- C++ -*------x---------------------------------------------------------
// $Id: Histogram3D.h,v 1.1 2006/10/03 17:10:40 bindewae Exp $
//
// Class:           Histogram
// 
// Base class:      -
//
// Derived classes: - 
//
// Author:          Eckart Bindewald
//
// Description:     This class implements the concept of a histogram.
// 
// -----------------x-------------------x-------------------x-----------------

#ifndef __Histogram3D_H__
#define __Histogram3D_H__

// Includes

#include <iostream>

#include <debug.h>
#include <Vec.h>

class Histogram3D {

public:

  /* CONSTRUCTORS */

  typedef Vec<unsigned int>::size_type size_type;
  typedef Vec<unsigned int>::size_type index_type;
  typedef Vec<Vec<Vec<double> > > data_type;

  Histogram3D();

  Histogram3D(size_type nx, double _minx, double _stepx,
	      size_type ny, double _miny, double _stepy,
	      size_type nz, double _minz, double _stepz) : minx(_minx), stepx(_stepx),
							 miny(_miny), stepy(_stepy),
							 minz(_minz), stepz(_stepz) {   
    histData = Vec<Vec<Vec<double> > >(nx, Vec<Vec<double> >(ny, Vec<double>(nz, 0.0) ) );
  }

  Histogram3D(const Histogram3D& orig);

  virtual ~Histogram3D();

  /* OPERATORS */

  /** Assigment operator. */
  Histogram3D& operator = (const Histogram3D& orig);
  /** comparison operator */
  friend bool operator==(const Histogram3D& left, const Histogram3D& right);
  /** output operator */
  friend ostream& operator << (ostream& os, const Histogram3D& object);
  /** input operator  */
  friend istream& operator >> (istream& is, Histogram3D& object);

  /* PREDICATES */

  /** return left side x-value of first bin */
  double getMinX() const { return minx; }
  /** return left side x-value of first bin */
  double getMinY() const { return miny; }
  /** return left side x-value of first bin */
  double getMinZ() const { return minz; }

  /** return RIGHT side x-value of last bin */
  double getMaxX() const { return minx + stepx * histData.size(); }
  /** return RIGHT side x-value of last bin */
  double getMaxY() const { return miny + stepy * histData[0].size(); }
  /** return RIGHT side x-value of last bin */
  double getMaxZ() const { return minz + stepz * histData[0][0].size(); }

  /** return bin heights */
  data_type getData() const { return histData; }

  /** return width of bins */
  double getStepX() const { return stepx; }

  /** return width of bins */
  double getStepY() const { return stepy; }

  /** return width of bins */
  double getStepZ() const { return stepz; }

  /** get the bin height which corresponds to the correct bin of value */
  double getValue(double x, double y, double z) const;

  /** get the bin height which corresponds to the correct bin of value */
  double getSafeValue(double x, double y, double z) const;

  /** get the bin height which corresponds to the correct bin of value */
  double getValue(const Vec<double>& v) const { PRECOND(v.size() == 3); return getValue(v[0], v[1], v[2]); }

  /** get the bin height which corresponds to the correct bin of value */
  double getSafeValue(const Vec<double>& v) const { PRECOND(v.size() == 3); return getSafeValue(v[0], v[1], v[2]); }

//   /** Interpolate between bins. No interpolation, 
//       if value is at middle of bin */
//   double getInterpolatedValue(double orig) const;

  /** return height of bin # d */
  double getBinValue(index_type nx, 
		     index_type ny,
		     index_type nz) const;

  /** return x-value of left side of bin */
  double getBinXStart(unsigned int d) const { return minx + (stepx * d); }
  /** return x-value of left side of bin */
  double getBinYStart(unsigned int d) const { return miny + (stepy * d); }
  /** return x-value of left side of bin */
  double getBinZStart(unsigned int d) const { return minz + (stepz * d); }

  size_type getDimX() const { return histData.size(); }
  size_type getDimY() const { return histData[0].size(); }
  size_type getDimZ() const { return histData[0][0].size(); }

  /** return x-value of left side of bin */
  double getXBinMiddle(unsigned int d) const { 
    return minx + (stepx * (d+0.5)); }
  /** return x-value of left side of bin */
  double getYBinMiddle(unsigned int d) const { 
    return miny + (stepy * (d+0.5)); }
  /** return x-value of left side of bin */
  double getZBinMiddle(unsigned int d) const { 
    return minz + (stepz * (d+0.5)); }

  bool isDefinedX(double x) const {
    int bin = getBinX(x);
    return ((bin >= 0) && (bin < static_cast<int>(getDimX())));
  }
  bool isDefinedY(double x) const {
    int bin = getBinY(x);
    return ((bin >= 0) && (bin < static_cast<int>(getDimY())));
  }
  bool isDefinedZ(double z) const {
    int bin = getBinZ(z);
    return ((bin >= 0) && (bin < static_cast<int>(getDimZ())));
  }

  bool isDefined(double x, double y, double z) const {
    return isDefinedX(x) && isDefinedY(y) && isDefinedZ(z);
  }

  bool isDefined(const Vec<double>& v) const {
    return isDefined(v[0], v[1], v[2]);
  }

  /** return bin number which corresponds to the value */
  int getBinX(double d) const;
  /** return bin number which corresponds to the value */
  int getBinY(double d) const;
  /** return bin number which corresponds to the value */
  int getBinZ(double d) const;
  /** return bin number which corresponds to the value */

  int getSafeBinX(double d) const;
  /** return bin number which corresponds to the value */
  int getSafeBinY(double d) const;
  /** return bin number which corresponds to the value */
  int getSafeBinZ(double d) const;

  /* MODIFIERS */

  /** set overall start value */
  void setMinX(double x) { minx = x; }
  /** set overall start value */
  void setMinY(double x) { miny = x; }
  /** set overall start value */
  void setMinZ(double x) { minz = x; }

  /** add data element */
  void addData(double x, double y, double z) { 
//     int bin = getBin(d);
//     if ((bin >= 0) && (bin < static_cast<int>(histData.size()))) {
//       histData[bin] += 1;
//     }
    ASSERT(false);
  }


  /** set data vector */
  void setData(const data_type& d) { histData = d; }

  /** set step size */
  void setStepX(double orig) { stepx = orig; }
  /** set step size */
  void setStepY(double orig) { stepy = orig; }
  /** set step size */
  void setStepZ(double orig) { stepz = orig; }

protected:
  /* OPERATORS  */
  /* PREDICATES */
  virtual bool compare(const Histogram3D& other) const;
  /* MODIFIERS  */
  virtual void copy(const Histogram3D& orig);

private:
  /* OPERATORS  */
  /* PREDICATES */
  /* MODIFIERS  */

private:
  /* PRIVATE ATTRIBUTES */
  data_type histData;
  double minx;
  double stepx;
  double miny;
  double stepy;
  double minz;
  double stepz;

}; // class Histogram3D

#endif /* __A_CLASS_H__ */

