// --*- C++ -*------x---------------------------------------------------------
// $Id: KnnNet.h,v 1.1.1.1 2006/07/03 14:43:20 bindewae Exp $
//
// Class:           KnnNet
// 
// Base class:      -
//
// Derived classes: - 
//
// Author:          Eckart Bindewald
//
// Description:     Network of knn classifiers
// 
// Reviewed by:     -
// -----------------x-------------------x-------------------x-----------------

#ifndef __KNN_NET_H__
#define __KNN_NET_H__

// Includes

#include <iostream>
#include <Vec.h>
#include <KnnNode.h>
#include <debug.h>

/** Implements network of k-nearest neigbor classifiers.
    @author Eckart Bindewald
    @see    Modern Applied Statistics with S / Ripley B
    @review */
class KnnNet : public ClassifierBase {
public:

  enum { KNNNET_LEVELS_MAX = 16, KNNNET_LEVELNODES_MAX = 10000 };

  KnnNet();

  KnnNet(const KnnNet& orig);

  virtual ~KnnNet();

  /* OPERATORS */

  /** Assigment operator. */
  KnnNet& operator = (const KnnNet& orig);

  friend ostream& operator << (ostream& os, const KnnNet& rval);
    
  friend istream& operator >> (istream& is, KnnNet& rval);

  /* PREDICATES */

  /** Is current state valid? */
  virtual bool isValid() const;

  /** How big is object? */
  virtual unsigned int size() const;

  /** central prediction method. Returns probability of each class according to 
      vote of classes of k nearest neighbors */
  virtual Vec<double> predictClassProb(const Vec<double>& v) const;

  /** central prediction method. Returns probability of each class according to 
      vote of classes of k nearest neighbors */
  virtual Vec<double> predictClassProb(const Vec<double>& v, unsigned int knownClass) const;

  /** central prediction method. Returns probability of each class according to 
      vote of classes of k nearest neighbors */
  virtual Vec<double> predictClassProbLevel(const Vec<double>& v, unsigned int level) const;

  /** central prediction method. Returns probability of each class according to 
      vote of classes of k nearest neighbors
      Supply known answer in order to write to training file.
  */
  virtual Vec<double> predictClassProbTrain(const Vec<double>& v, unsigned int trueClass,
				    ostream& writeFile) const;

  virtual Vec<double> getLastPrediction() const { return lastPrediction; }

  /** get predictions of all nodes for result class dim */
  virtual Vec<double> getLastNodePredictions(unsigned int dim) const;

  /** returns number of levels. */
  virtual unsigned int getNumLevels() const { return numLevels; }

  /** return set filename for writing intermediate training data */
  virtual const string& getWriteFileName() const { return writeFileName; }

  /** return level for writing intermediate training data : -1: no writing, 0: write after read data, 1: write after first level etc */
  virtual int getWriteLevel() const { return writeLevel; }

  /** write intermediate feature vector for all training vector */
  virtual void writeLevelTrainVectors(ostream& writeFile, int numEntries) const;
  
  /** write parameter file */
  virtual void write(ostream& os) const;

  /** start statistics mode */
  virtual void startStatistics() {
    for (unsigned int i = 0; i <  knnNodes.size(); ++i) {
      knnNodes[i].startStatistics();
    }
  }

  /** returns usage histogram. Call startStatistics first.  */
  virtual Vec<Vec<double> > getUsageHistogram(unsigned int classId) const {
    Vec<Vec<double> > result(knnNodes.size());
    for (unsigned int i = 0; i < result.size(); ++i) {
      Vec<Vec<double> > hist = knnNodes[i].getUsageHistogram();
      if (classId < hist.size()) {
	result[i] = hist[classId];
      }
    }
    return result;
  }

  virtual const KnnNode& getNode(unsigned int n) const { return knnNodes[n]; }

  virtual const Vec<KnnNode>& getNodes() const { return knnNodes; }

  virtual Vec<unsigned int> getNumClusters() const {
    Vec<unsigned int> result(knnNodes.size());
    for (unsigned int i = 0; i < result.size(); ++i) {
      result[i] = knnNodes[i].getNumClusters();
    }
    return result;
  }

  virtual int getVerboseLevel() const { return verboseLevel; }

  virtual void saveData(string saveDir) const;

  /** get user defined properties */
  virtual const Vec<double>& getProperties() const { return properties; }


  /* MODIFIERS */
  
  virtual void clear() {
    numLevels = 0;
    knnNodes.clear();
    knnLevelIds.clear();
    levelMasks.clear();
    levelTargetDims.clear();
    knnLevelTargetIds.clear();
    knnLevelTargetDim.clear();
    knnLevelMultipliers.clear();
  }

  virtual KnnNode& getNode2(unsigned int n) { return knnNodes[n]; }

  /** optimize scaling of node using simple Monte Carlo steps */
  virtual Vec<double> optimizeScaling(unsigned int knnId,
			      int numSteps,
			      int verboseLevel,
			      double stepWidth,
			      unsigned int numTrials);

  /** optimize scaling of all nodes using simple Monte Carlo steps */
  virtual void  optimizeScaling(      int numSteps,
			      int verboseLevel,
			      double stepWidth,
			      unsigned int numTrials);

  /** returns level of node */
  virtual unsigned int nodeLevel(unsigned int nodeId) const;

  /** returns all nodes of level */
  virtual Vec<unsigned int> findLevelNodes(unsigned int level) const;
  
  /** optimize scaling of node using simple Monte Carlo steps */
  virtual void bootstrapLevelNode(const Vec<Vec<double> >& levelData,
			  const Vec<unsigned int>& levelClasses,
			  unsigned int knnId,
			  int numSteps,
			  int knnThin,	  
			  int verboseLevel,
			  double stepWidth,
			  unsigned int numTrials,
			  unsigned int simMax,
			  double cutoff,
			  const string& writeDir);
  
  /** optimize scaling of node using simple Monte Carlo steps */
  virtual void bootstrapLevel(const Vec<Vec<double> >& levelData,
		      const Vec<unsigned int>& levelClasses,
		      unsigned int level,
		      int numSteps,
		      int knnThin,
		      int verboseLevel,
		       double stepWidth,
		      unsigned int numTrials,
		      unsigned int simMax,
		      double cutoff,
		      const string& writeDir);

  /** optimize scaling of node using simple Monte Carlo steps */
  virtual void bootstrap(const Vec<Vec<double> >& rawData,
		 const Vec<unsigned int>& rawClasses,
		 int numSteps,
		 int knnThin,
		 int verboseLevel,
		 double stepWidth,
		 unsigned int numTrials,
		 unsigned int simMax,
		 double cutoff,
		 const string& writeDir);
  
  virtual void read(istream& is, string dataDir);

  virtual void recluster() {
    for (unsigned int i = 0; i < knnNodes.size(); ++i) {
      knnNodes[i].recluster();
    }
  }

  /** sets cutoff for simpleRepresentativeLinkage */
  virtual void setClusterCutoff(double x) {
    clusterCutoff = x;
    for (unsigned int i = 0; i < knnNodes.size(); ++i) {
      knnNodes[i].setClusterCutoff(x);
    }
  }

  /** sets cutoff for simpleRepresentativeLinkage */
  virtual void setClusterCutoff2(double x) {
    clusterCutoff2 = x;
    for (unsigned int i = 0; i < knnNodes.size(); ++i) {
      knnNodes[i].setClusterCutoff2(x);
    }
  }

  /** sets number of classes of all knn nodes */
  virtual void setGaussDev(double g) {
    for (unsigned int i = 0; i < knnNodes.size(); ++i) {
      knnNodes[i].setGaussDev(g);
    }
  }

  /** sets number of classes of all knn nodes */
  virtual void setNumClasses(unsigned int num) {
    for (unsigned int i = 0; i < knnNodes.size(); ++i) {
      knnNodes[i].setNumClasses(num);
    }
  }

  /** set user defined properties */
  virtual void setProperties(const Vec<double>& v) { properties = v; }

  virtual void setVerboseLevel(int n) { 
    verboseLevel = n;
    for (unsigned int i = 0; i < knnNodes.size(); ++i) {
      knnNodes[i].setVerboseLevel(n);
    }
  }

  /** return set filename for writing intermediate training data */
  virtual void setWriteFileName(const string& s) { writeFileName = s; }

  /** return level for writing intermediate training data : -1: no writing, 0: write after read data, 1: write after first level etc */
  virtual void setWriteLevel(int n) { writeLevel = n; }

  virtual void setNoSelfMode(bool b) {
    for (unsigned int i = 0; i < knnNodes.size(); ++i) {
      knnNodes[i].setNoSelfMode(b);
    }
  }

  /** apply thinning operation to all nodes */
  virtual void thin(unsigned int thinK);


protected:
  /* OPERATORS  */
  /* PREDICATES */
  /* MODIFIERS  */
  virtual void copy(const KnnNet& other);

private:
  /* OPERATORS  */
  /* PREDICATES */
  /* MODIFIERS  */

private:
  /* PRIVATE ATTRIBUTES */

  unsigned int numLevels; // number of levels

  /** verbose level */
  int verboseLevel;
  int writeLevel;
  double clusterCutoff;
  double clusterCutoff2;
  string writeFileName;

  Vec<double> properties;
  Vec<KnnNode> knnNodes; // library of knn
  Vec<Vec<unsigned int> > knnOrigMasks; // what is data read mask of j'th node
  Vec<string> knnOrigFileNames; // what is data file name of j'th node?
  Vec<Vec<int> > knnLevelIds; // for i'th level, what is id (index) of j'th knn?
  Vec<Vec<Vec<unsigned int> > > levelMasks; // for i'th level, what is mask for j'th knn?
  Vec<unsigned int> levelTargetDims; // for i'th level, what is dimension of target feature vector
  Vec<Vec<unsigned int> > knnLevelTargetIds; // for i'th level, what is target index of j'th knn?
  Vec<Vec<unsigned int> > knnLevelTargetDim; // for i'th level, which dimension of result of j'th knn should be used?
  Vec<Vec<double> > knnLevelMultipliers; // weight for i'th level, j'th knn

  mutable Vec<double> lastPrediction;

};

#endif /* __ACLASS_H__ */

