// --*- C++ -*------x---------------------------------------------------------
// $Id: editcube4d.cc,v 1.1 2006/10/18 12:08:30 bindewae Exp $
//
// Program:         - 
//
// Author:          Eckart Bindewald
//
// Project name:    -
//
// Date:            $Date: 2006/10/18 12:08:30 $
//
// Description:     
// 
// -----------------x-------------------x-------------------x-----------------

#include <iostream>
#include <fstream>
#include <string>
#include <Vec.h>
#include <debug.h>
#include <GetArg.h>
#include <FileName.h>
#include <vectornumerics.h>
#include <Random.h>
#include <Histogram4D.h>

void
helpOutput(ostream& os)
{
  os << "usage: editcube -i inputfile --of outputformat --subset vector --verbose n" << endl;
  os << "other options: " << endl
     << "--div divcubefilename : divide matrix elements by this cube" << endl
     << "--reverse-columns 0|1 " << endl
     << "--reverse-rows 0|1 " << endl;
}

/** output of command line parameter with which the program was called. */
void
parameterOutput(ostream& os, int argc, char** argv)
{
  for (int i = 0; i < argc; i++)
    {
      os << argv[i] << " ";
    }
  os << endl;
}

/** reads matrix in format that is also used by program "R" */
/*
Vec<Vec<double> >
readPlainMatrix(istream& is)
{
  Vec<Vec<double > > result;
  while (is) {
    string line = getLine(is);
    Vec<string> words = getTokens(line);
    if (words.size() > 0) {
      Vec<double> row(words.size());
      for (unsigned int i = 0; i < words.size(); ++i) {
	row[i] = stod(words[i]);
      }
      result.push_back(row);
    }
  }
  return result;
}
*/

void
generateKnnPoints(ostream& os, 
		  int knnNum, 
		  double prob,
		  double posx, 
		  double posy, 
		  double deltax,
		  double deltay)
{
  Random& rnd = Random::getInstance();
  double px, py;
  for (int i = 0; i < knnNum; ++i) {
    px = posx + (rnd.getRandf()-0.5)*deltax;
    py = posy + (rnd.getRandf()-0.5)*deltay;
    os << px << " " << py << " ";
    if (rnd.getRandf() < prob) {
      os << "1";
    }
    else {
      os << "0";
    }
    os << endl;
  }
}

void
generateKnnPoints(ostream& os, 
		  int knnNum, 
		  double prob,
		  double posx, 
		  double posy, 
		  double posz,
		  double deltax,
		  double deltay,
		  double deltaz)
{
  Random& rnd = Random::getInstance();
  double px, py, pz;
  for (int i = 0; i < knnNum; ++i) {
    px = posx + (rnd.getRandf()-0.5)*deltax;
    py = posy + (rnd.getRandf()-0.5)*deltay;
    pz = posz + (rnd.getRandf()-0.5)*deltaz;
    os << px << " " << py << " " << pz << " ";
    if (rnd.getRandf() < prob) {
      os << "1";
    }
    else {
      os << "0";
    }
    os << endl;
  }
}


void
generateKnnPoints(ostream& os, 
		  int knnNum, 
		  const Vec<Vec<double> >& matrix, 
		  double minx, 
		  double miny, 
		  double deltax,
		  double deltay)
{
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    for (unsigned int j = 0; j < matrix[i].size(); ++j) {
      generateKnnPoints(os, knnNum, matrix[i][j],  i * deltax + minx, j * deltay + miny, deltax, deltay);
    }
  }
}

void
generateKnnPoints(ostream& os, 
		  int knnNum, 
		  const Vec<Vec<Vec<double> > > & cube,
		  double minx, 
		  double miny, 
		  double minz, 
		  double deltax,
		  double deltay,
		  double deltaz)
{
  for (unsigned int i = 0; i < cube.size(); ++i) {
    for (unsigned int j = 0; j < cube[i].size(); ++j) {
      for (unsigned int k = 0; k < cube[i][j].size(); ++k) {
	generateKnnPoints(os, knnNum, cube[i][j][k],  i * deltax + minx, j * deltay + miny, k * deltaz + minz, 
			  deltax, deltay, deltaz);
      }
    }
  }
}

Vec<Vec<Vec<double> > >
readListCube(istream& is)
{
  string line = getLine(is);
  Vec<string> words = getTokens(line);
  ERROR_IF(words.size() != 3,
	   "Cube dimensions expected in first line!");
  unsigned dx = stoui(words[0]);
  unsigned dy = stoui(words[1]);
  unsigned dz = stoui(words[2]);
  ERROR_IF((dx == 0) || (dy == 0) || (dz == 0),
	   "Cube dimensions have to be larger zero!");
  Vec<Vec<Vec<double> > > result(dx, Vec<Vec<double> >(dy, Vec<double>(dz, 0.0)));
  while (is) {
    line = getLine(is);
    words = getTokens(line);
    if (words.size() == 0) {
      continue;
    }
    ERROR_IF(words.size() != 4,
	     "Bad data line, 4 entries expected!");
    unsigned int ix = stoui(words[0]);
    unsigned int iy = stoui(words[1]);
    unsigned int iz = stoui(words[2]);
    double val = stod(words[3]);
    result[ix][iy][iz] = val;
  }
  return result;
}

Vec<Vec<double> >
readListMatrix(istream& is)
{
  string line = getLine(is);
  Vec<string> words = getTokens(line);
  ERROR_IF(words.size() != 2,
	   "Matrix dimenstions expected in first line!");
  unsigned dx = stoui(words[0]);
  unsigned dy = stoui(words[1]);
  ERROR_IF((dx == 0) || (dy == 0),
	   "Matrix dimensions have to be larger zero!");
  Vec<Vec<double> > result(dx, Vec<double>(dy, 0.0));
  while (is) {
    line = getLine(is);
    words = getTokens(line);
    if (words.size() == 0) {
      continue;
    }
    ERROR_IF(words.size() != 3,
	     "Bad data line, 3 entries expected!");
    unsigned int ix = stoui(words[0]);
    unsigned int iy = stoui(words[1]);
    double val = stod(words[2]);
    result[ix][iy] = val;
  }
  return result;
}


void
monotomizeFunction(Vec<double>& v)
{
  double last = v[0];
  for (unsigned int i = 1; i < v.size(); ++i) {
    if (v[i] < last) {
      v[i] = last;
    }
    else {
      last = v[i];
    }
  }
}
void
monotomizeFunctionDown(Vec<double>& v)
{
  if (v.size() < 2) {
    return;
  }
  double last = v[v.size()-1];
  for (int i = v.size()-2; i >= 0; --i) {
    if (v[i] > last) {
      v[i] = last;
    }
    else {
      last = v[i];
    }
  }
}

void
monotomizeMatrixRows(Vec<Vec<double> >& matrix)
{
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    monotomizeFunction(matrix[i]);
  }
}

void
monotomizeMatrixRowsDown(Vec<Vec<double> >& matrix)
{
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    monotomizeFunctionDown(matrix[i]);
  }
}

void
monotomizeMatrixColumns(Vec<Vec<double> >& matrix)
{
  for (unsigned int i = 0; i < matrix[0].size(); ++i) {
    Vec<double> col = getColumn(matrix, i);
    monotomizeFunction(col);
    setColumn(matrix, col, i);
  }
}

void
monotomizeMatrixColumnsDown(Vec<Vec<double> >& matrix)
{
  for (unsigned int i = 0; i < matrix[0].size(); ++i) {
    Vec<double> col = getColumn(matrix, i);
    monotomizeFunctionDown(col);
    setColumn(matrix, col, i);
  }
}

bool
isMonoton(const Vec<double>& v, double err)
{
  for (unsigned int i = 1; i < v.size(); ++i) {
    if (v[i] + err < v[i-1]) {
      return false;
    }
  }
  return true;
}

bool
isMonoton(const Vec<Vec<double> >& v, double err)
{
  for (unsigned int i = 0; i < v.size(); ++i) {
    if (!isMonoton(v[i], err)) {
      return false;
    }
  }
  for (unsigned int i = 0; i < v[0].size(); ++i) {
    if (!isMonoton(getColumn(v,i), err)) {
      return false;
    }
  }
  return true;
}

bool
isMonoton(const Vec<Vec<Vec<double> > >& v, double err)
{
  for (unsigned int i = 0; i < v.size(); ++i) {
    if ((!isMonoton(v[i], err))
	|| (!isMonoton(getYSlice(v, i), err))
	|| (!isMonoton(getZSlice(v, i), err))) {
      return false;
    }
  }
  return true;
}



void
monotonizeMatrix(Vec<Vec<double> >& matrix, double err)
{
  int iter = 0;
  do {
    Vec<Vec<Vec<double> > > matrices(4, matrix);
    double norm = 1.0 / static_cast<double>(matrices.size()); 
    monotomizeMatrixRows(matrices[0]);
    monotomizeMatrixRowsDown(matrices[1]);
    monotomizeMatrixColumns(matrices[2]);
    monotomizeMatrixColumnsDown(matrices[3]);
    for (unsigned int i = 0; i < matrix.size(); ++i) {
      for (unsigned int j = 0; j < matrix[i].size(); ++j) {
	matrix[i][j] = 0.0;
	for (unsigned int k = 0; k < 4; ++k) {
	  matrix[i][j] += matrices[k][i][j];
	}
	matrix[i][j] *= norm;
      }
    }
    if (iter > 1000000) {
      cout << "Could not find monoton matrix!" << endl;
      break;
    }
  }
  while (!isMonoton(matrix, err));
}


void
monotonizeCube(Vec<Vec<Vec<double> > >& cube, double err)
{
  int iter = 0;
  do {
    cerr << "monotonize cube iter " << iter++ + 1 << endl;
    Vec<Vec<Vec<Vec<double> > > > cubes(3, cube);
    unsigned int nn = cube.size();
    double norm = 1.0 / 3.0;
    for (unsigned int i = 0; i < nn; ++i) {
      monotonizeMatrix(cubes[0][i], err);
      Vec<Vec<double> > slice = getYSlice(cubes[1], i);
      monotonizeMatrix(slice, err);
      setYSlice(cubes[1], slice, i);
      slice = getZSlice(cubes[2], i);
      monotonizeMatrix(slice, err);
      setZSlice(cubes[2], slice, i);
    }
    for (unsigned int i = 0; i < cube.size(); ++i) {
      for (unsigned int j = 0; j < cube[i].size(); ++j) {
	for (unsigned int k = 0; k < cube[i][j].size(); ++k) {
	  cube[i][j][k] = 0.0;
	  for (unsigned int m = 0; m < 3; ++m) {
	    cube[i][j][k] += cubes[m][i][j][k];
	  }
	  cube[i][j][k] *= norm;
	}
      }
    }
  }
  while (!isMonoton(cube, err));
}

/** monotonize 3d sub cubes, leave first dimension alone */
void
monotonizeSubCubes(Vec<Vec<Vec<Vec<double> > > > & cube4d, double err)
{
  for (unsigned int i = 0; i < cube4d.size(); ++i) {
    monotonizeCube(cube4d[i], err);
  }
}

void
writeMatrix(ostream& os,
	    const Vec<Vec<double> >& matrix) {
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    for (unsigned int j = 0; j < matrix[i].size(); ++j) {
      os << matrix[i][j] << " ";
    }
    os << endl;
  }
}

void
writeSliceFiles(const Vec<Vec<Vec<double> > >& cube, 
		const string& inputFileName)
{
  for (unsigned int i = 0; i < cube.size(); ++i) {
    string fileName = inputFileName + "." + uitos(i) + ".matrix";
    ofstream oFile(fileName.c_str());
    ERROR_IF(!oFile, "Error opening output file!");
    writeMatrix(oFile, cube[i]);
    oFile.close();
  }
}

Vec<Vec<Vec<Vec<double> > > >
readCube4D(istream& inputFile,
	 int inputFormat) {
  Vec<Vec<Vec<Vec<double> > > > cube;
  Histogram4D histogram;
  switch (inputFormat) {
  case 1:
    ERROR("Sorry, reading of 4D cube as list is not yet implemented!");
    // cube = readListCube(inputFile);
    break;
  case 2: 
    inputFile >> histogram;
    cube = histogram.getData();
    break;
  case 3:
    // cube = readPlainCube(inputFile);
    inputFile >> cube;
    break;
  default: ERROR("Unknown cube read mode!");
  }
  return cube;
}

int
main(int argc, char ** argv)
{

  bool helpMode;
  int argcFile = 0;
  char ** argvFile = 0;
  int inputFormat = 3;
  int knnNum = 0;
  int monotonizeMode = 0;
  int outputFormat = 3;
  int reverseColumnMode = 0;
  int reverseRowMode = 0;
  int transposeMode = 0;
  double deltax = 0.1;
  double deltay = 0.1;
  double deltaz = 0.1;
  double err = 0.01;
  double minx = 0.0;
  double miny = 0.0;
  double minz = 0.0;
  unsigned int verboseLevel = 1;
  string commandFileName;
  string divFileName;
  string subsetString;
  string logFileName; //  = "mainprogramtemplate.log";
  string rootDir = ".";
  string inputFileName;
  double limitVal = 0.0;
  Vec<unsigned int> subset;

  getArg("-help", helpMode, argc, argv);

  if ((argc < 2) || helpMode)  {
    helpOutput(cout);
    exit(0);
  }

  getArg("-root", rootDir, argc, argv, rootDir);
  addSlash(rootDir);

  getArg("-commands", commandFileName, argc, argv, commandFileName);
  addPathIfRelative(commandFileName, rootDir);

  if (commandFileName.size() > 0) {
    ifstream commandFile(commandFileName.c_str());
    if (!commandFile) {
      if (isPresent("-commands", argc, argv)) {
	ERROR_IF(!commandFile, "Error opening command file.");
      }
      else {
	cerr << "Warning: Could not find command file: " + commandFileName 
	     << endl;
      }
    }
    else {
      argvFile = streamToCommands(commandFile, argcFile, 
				  string("mainprogramtemplate"));
    }
    commandFile.close();
  }

  getArg("-div", divFileName, argc, argv, divFileName);
  getArg("i", inputFileName, argc, argv);
  getArg("l", limitVal, argc, argv, limitVal);
  getArg("-if", inputFormat, argcFile, argvFile, inputFormat);
  getArg("-if", inputFormat, argc, argv, inputFormat);
  getArg("-log", logFileName, argc, argv, logFileName);
  getArg("-log", logFileName, argcFile, argvFile, logFileName);
  addPathIfRelative(logFileName, rootDir);
  getArg("-knn", knnNum, argc, argv, knnNum);
  getArg("-monotonize", monotonizeMode, argc, argv, monotonizeMode);
  getArg("-minx", minx, argc, argv, minx);
  getArg("-miny", miny, argc, argv, miny);
  getArg("-deltax", deltax, argc, argv, deltax);
  getArg("-deltay", deltay, argc, argv, deltay);
  getArg("-of", outputFormat, argcFile, argvFile, outputFormat);
  getArg("-of", outputFormat, argc, argv, outputFormat);
  getArg("-subset", subsetString, argc, argv, subsetString);
  getArg("-transpose", transposeMode, argc, argv, transposeMode);
  subset = parseStringToVector(subsetString);
  convert2InternalCounting(subset);
  sort(subset.begin(), subset.end());
  getArg("-reverse-columns", reverseColumnMode, argc, argv, reverseColumnMode);
  getArg("-reverse-rows", reverseRowMode, argc, argv, reverseRowMode);
  getArg("-verbose", verboseLevel, argcFile, argvFile, verboseLevel);
  getArg("-verbose", verboseLevel, argc, argv, verboseLevel);


  if (logFileName.size() > 0) {
    ofstream logFile(logFileName.c_str(), ios::app);
    parameterOutput(logFile, argc, argv);
    if (argcFile > 1) {
      logFile << "Parameters from command file: ";
      parameterOutput(logFile, argcFile, argvFile);
    }
    logFile.close();
  }


  /***************** MAIN PROGRAM *****************************/


  Vec<Vec<Vec<Vec<double> > > > cube, divCube;
  ifstream inputFile(inputFileName.c_str());
  ERROR_IF(!inputFile, "Error opening input file!");
  cube = readCube4D(inputFile, inputFormat);
  inputFile.close();
  ERROR_IF(cube.size() == 0, "No cube defined!");

  if (divFileName.size() > 0) {
    ifstream divFile(divFileName.c_str());
    ERROR_IF(!divFile, "Error opening input file!");
    divCube = readCube4D(divFile, inputFormat);
    divFile.close();
    ERROR_IF(divCube.size() == 0, "No cube defined!");
    for (unsigned int i = 0; i < cube.size(); ++i) {
      for (unsigned int j = 0; j < cube[i].size(); ++j) {
	for (unsigned int k = 0; k < cube[i][j].size(); ++k) {
	  for (unsigned int m = 0; m < cube[i][j][k].size(); ++m) {
	    if (divCube[i][j][k][m] != 0) {
	      cube[i][j][k][m] /= divCube[i][j][k][m];
	    }
	    else {
	      cube[i][j][k][m] = 0.0;
	    }
	  }
	}
      }
    }
  }

//   if (transposeMode) {
//     cout << "Transposing cube!" << endl;
//     cubeTranspose(cube);
//   }

//   if (subset.size() > 0) {
//     cube = getCubeSubset(cube, subset);
//     ERROR_IF(cube.size() != cube[0].size(),
// 	     "Result cube of subset is not quadratic!");
//   }

//   if (reverseRowMode) {
//     reverse(cube.begin(), cube.end());
//   }
//   if (reverseColumnMode) {
//     for (unsigned int i = 0; i < cube.size(); ++i) {
//       reverse(cube[i].begin(), cube[i].end());
//     }
//   }

  if (monotonizeMode) {
    if (verboseLevel > 0) {
      cout << "Monotimizing cube!" << endl;
    }
    // ERROR("Sorry, monotonizing is currently not available!");
    monotonizeSubCubes(cube, err);
  }

  switch (outputFormat) {
  case 0:
    break; // do nothing
  case 1:
    for (unsigned int i = 0; i < cube.size(); ++i) {
      for (unsigned int j = 0; j < cube[i].size(); ++j) {
	for (unsigned int k = 0; k < cube[i][j].size(); ++k) {
	  for (unsigned int m = 0; m < cube[i][j][k].size(); ++m) {
	    if (cube[i][j][k][m] > limitVal) {
	      cout << i + 1 << " " << j + 1 << " " << k + 1 << " " << m+1 << " " 
		   << cube[i][j][k][m] << endl;
	    }
	  }
	}
      }
    }
    break;
    case 3:
    for (unsigned int i = 0; i < cube.size(); ++i) {
      for (unsigned int j = 0; j < cube[i].size(); ++j) {
	for (unsigned int k = 0; k < cube[i][j].size(); ++k) {
	  for (unsigned int m = 0; m < cube[i][j][k].size(); ++m) {
	    cout << cube[i][j][k][m] << " ";
	  }
	  cout << endl;
	}
	cout << endl;
      }
      cout << endl;
    }
    break;
  case 4:
    for (unsigned int i = 0; i < cube.size(); ++i) {
      for (unsigned int j = 0; j <= i; ++j) {
	for (unsigned int k = 0; k <= j; ++k) {
	  for (unsigned int m = 0; m <= k; ++m) {
	  if (cube[i][j][k][m] > limitVal) {
	    cout << i + 1 << " " << j + 1 << " " << k + 1 << " " << m+1 << " " 
		 << cube[i][j][k][m] << endl;
	  }
	  }
	}
      }
    }
    break;
  case 6:
    ERROR("Sorry, output format 6 not yet implemented");
    for (unsigned int i = 0; i < cube.size(); ++i) {
      for (unsigned int j = 0; j < cube[i].size(); ++j) {
	for (unsigned int k = 0; k < cube[i][j].size(); ++k) {
	  // if (cube[i][j] > limitVal) {
	  cout << (deltax*i  + minx) << " " << (deltay*j + miny) << " " << deltaz*k + minz << " " 
	       << cube[i][j][k] << endl;
	  // }
	}
      }
    }
    break;
  case 7:
    cout << cube << endl;
    break;
  case 8:
    ERROR("Sorry, output format 8 not yet implemented");
    // writeSliceFiles(cube, inputFileName);
    break;
  default:
    ERROR("Unknown cube output format!");
  }

//   if (knnNum > 0) {
//     generateKnnPoints(cout, knnNum, cube, minx, miny, minz, deltax, deltay, deltaz);    
//   }
    
  return 0;
}
