
#define __STL_NO_DRAND48 

#include <math.h>
#include <Vec.h>
#include <algorithm>
#include <Random.h>
#include <vectornumerics.h>

/** vector with probabilities of different entities.
    values must be between 0 and 1
*/
double entropyShannon(const Vec<double>& v); 

/** 
 * @param v: arbitrary vector with double values.
 * from that the routine computes a normalized histogram and using that the Shannon entropy
*/
double
entropyOfHistogram(const Vec<double>& v);

/** convert vector to probabilities */
template<class T>
inline
Vec<double>
getProbabilities(const Vec<T>& v, const Vec<T>& alphabet, 
		 unsigned int start, unsigned int end)
{
  Vec<double> result(alphabet.size(), 0.0);
  for (unsigned int i = 0; i < alphabet.size(); ++i) {
    unsigned int n = vecCount(v, alphabet[i], start, end);
    // cout << "Number of " << alphabet[i] << " : " << n << endl;
    result[i] = static_cast<double>(n) / static_cast<double>(v.size());
  }
  return result;
}

/** return index of alphabet letter, which occurs most frequent */
template<class T>
inline
unsigned int
mostFrequentLetter(const Vec<T>& v, const Vec<T>& alphabet)
{
  Vec<double> frequencies = getProbabilities(v, alphabet, 0, v.size());
  unsigned int result = findMaxIndex(frequencies);
  POSTCOND(result < alphabet.size());
  return result;
}

/** return Shannon's entropy measure for information */
template<class T>
inline
double
entropyShannon(const Vec<T>& v, const Vec<T>& alphabet,
	       unsigned int start, unsigned int end)
{
  Vec<double> prob = getProbabilities(v, alphabet, start, end);
  return entropyShannon(prob);
}

/** return Shannon's entropy measure for information
    group sizes are integers
 */
template<class T>
inline
double
entropyShannon(const Vec<T>& v, const Vec<T>& alphabet,
	       const Vec<unsigned int>& groups)
{
  unsigned int count = 0;
  double result = 0.0;
  for (unsigned int i = 0; i < groups.size(); ++i) {
    Vec<double> prob = getProbabilities(v, alphabet, count, count + groups[i]);
    result += entropyShannon(prob);  
    count += groups[i];
  }
  POSTCOND(count == v.size());
  return result;
}

/** return Shannon's entropy measure for information */
template<class T>
inline
double
computeExpectedEntropy(const Vec<T>& vOrig, const Vec<T>& alphabet,
		       const Vec<unsigned int>& groups,
		       double& stddev, double& stddevMean)
{
  const unsigned int maxIter = 1000;
  Vec<T> v = vOrig;
  double sum = 0.0;
  double sqSum = 0.0;
  for (unsigned int i = 0; i < maxIter; ++i) {
    random_shuffle(v.begin(), v.end());
    double ent = entropyShannon(v, alphabet, groups);
    sum += ent;
    sqSum += ent * ent;
  }
  double avg = sum / maxIter;
  stddev = sqrt(fabs(sqSum / maxIter - avg * avg));
  stddevMean = stddev / sqrt(maxIter);
  return avg;
}
	       
/** return Shannon's entropy measure for information */
template<class T>
inline
double
computeEntropyPositionalZScore(const Vec<T>& v, const Vec<T>& alphabet,
			       const Vec<unsigned int>& groups)
{
  double ent = entropyShannon(v, alphabet, groups);
  double stddev = 0.0;
  double stddevMean = 0.0;
  double expectedEnt = computeExpectedEntropy(v, alphabet, groups, stddev,
					      stddevMean);
  if (stddev <= 0.0) {
    return 0.0; // not clean: corresponds to a conserved position!?
  }
  return (ent - expectedEnt) / stddev;
}


/**
 * returns id of scored fragment according to Boltzmann statistics
 * assumes that fragment vec is ordered such that best (HIGHEST!) score is element zero
 * assumes that the score is store in fragments[i].first
 */
unsigned int
chooseBoltzmannElementHighestBest(const Vec<double>& scores, double kt);

/**
 * returns id of scored fragment according to Boltzmann statistics
 * assumes that fragment vec is ordered such that best (HIGHEST!) score is element zero
 * assumes that the score is store in fragments[i].first
 */
unsigned int
chooseBoltzmannElementLowestBest(const Vec<double>& scores, double kt);
