#include <generalNumerics.h>
#include <debug.h>
#include <math.h>
#include <realnumerics.h>

double
factorial(int n)
{
  double result = 1.0;
  for (int i = 2; i <= n; ++i) {
    result *= i;
  }
  POSTCOND(isDefined(result));
  return result;
}

double
logFactorial(int n)
{
  double result = 0.0;
  for (int i = 2; i <= n; ++i) {
    result += log(static_cast<double>(i));
  }
  POSTCOND(isDefined(result));
  return result;
}


/** returns poisson function value of observing k rare independent events if "expected" events are expected */
double poisson(int k, double expected) {
  return pow(expected, k) * exp(-expected) / factorial(k);
}

/** returns poisson function value of observing k rare independent events if "expected" events are expected */
double logPoisson(int k, double expected) {
  return (k*log(expected)) - expected - logFactorial(k);
}

/** returns poisson function value of observing k or less rare independent events if "expected" events are expected */
double poissonCdf(int k, double expected) {
  double sum = 0.0;
  for (int i = 0; i <= k; ++i) {
    sum += exp((i* log(expected)) - logFactorial(i));
  }
  double result = exp(-expected) * sum;
  if (result > 1.0) {
    result = 1.0;
  } else if (result < 0.0) {
    result = 0.0;
  }
  POSTCOND((result <= 1.0) && (result >= 0.0));
  return result;
}

/** how likely is it to observe this many or more counts?
 * same as probability as NOT observing less counts */
double poissonCdfP(int k, double expected) {
  double result = 1.0;
  if (k > 0) {
    result = 1.0 - poissonCdf(k - 1, expected);
  }
  ERROR_IF(result < -0.1 , "Internal error computing P value based on Poisson distribution. Obtained value less than -0.1");
  ERROR_IF(result > 1.1 , "Internal error computing P value based on Poisson distribution. Obtained value greater than 1.1");
  if (result < -0.1) {
    result = 0.0;
  }
  if (result > 1.1) {
    result = 1.0;
  }
  POSTCOND((result >= 0.0) || (result <= 1.0));
  return result;
}

double
nOverK(int n, int k) 
{
  ASSERT(n > 0);
  ASSERT(k > 0);
  ASSERT(n >= k);
//   if (k > n) {
//     return 0.0;
//   }
//   if (k == 1) {
//     return static_cast<double>(n);
//   }
//   if ((k == n) || (k == 0)) {
//     return 0.0;
//   }
  double result = (factorial(n) / factorial(k)) / factorial(n - k);
  POSTCOND(isDefined(result)); // avoid not-a-number (overflow) case
  return result;
}

double
logNOverK(int n, int k) 
{
  ASSERT(n > 0);
  ASSERT(k > 0);
  ASSERT(n >= k);
//   if (k > n) {
//     ASSERT(false);
//     return 0.0;
//   }
//   if (k == 1) {
//     return log(static_cast<double>(n));
//   }
//   if ((k == n) || (k == 0)) {
//     ASSERT(false);
//     return 0.0;
//   }
  double result = (logFactorial(n) - logFactorial(k)) - logFactorial(n - k);
  POSTCOND(isDefined(result)); // avoid not-a-number (overflow) case
  return result;
}

double
lg2(double x)
{
  const double LOG2 = log(2.0);
  if (x <= 0.0) {
    return 0.0;
  }
  return log(x) / LOG2;
}

/** returns logistic function */
double
logistic(double x)
{
  double result = (exp(x) / (1 + exp(x)));
  // ASSERT((result >= 0.0) && (result <= 1.0), exception);
  return result;
}


