#include <geom3D.h>

/** return min distance between point and line , defined trough offset and 
    direction
    positive result, if it projects to positive half line, negative otherwise
*/

double
distanceToLine(const Vector3D& point, 
	       const Vector3D& offset, 
	       const Vector3D& direction)
{
  PRECOND( (direction.length()>0.0));
 // (point!=direction)&&(offset!=direction)
  double result = 0.0;
  Vector3D dVec = point - offset;
  double dLength = dVec.length();
  if (dLength == 0.0) {
    return 0.0; // test point happens to be start point of line
  }
  double cosAlpha = dVec * direction /(dLength*direction.length());
  double alpha = acos(cosAlpha);
  result = dLength * sin(alpha);
  if (cosAlpha < 0.0)
    {
      result *= -1.0;
      ASSERT(result <= 0.0);
    }
  POSTCOND((result*cosAlpha) >= 0.0);
  return result;
} 

/** returns distance between two lines in 3D space defined
 * through x0+x*t and y0+y*s  .
 * It returns a 3D vector with the 1. component being 
 * "t", the second component being "s" and the third
 * component being the distance d 
 * Formulas from: Paper by Bard, Himel in class of Dr. Dennis Merino
 * (Mississipi College, USA)
 * www.mc.edu/campus/users/travis/maa/proceedings/spring2001/bard.himel.pdf  
 */

Vector3D 
computeLineDistance(const Vector3D& x0,
		    const Vector3D& x,
		    const Vector3D& y0, 
		    const Vector3D& y) {
  PRECOND(x.lengthSquare() > 0.0);
  PRECOND(y.lengthSquare() > 0.0);
  double A = x.lengthSquare();
  double B = 2.0 * ((x * x0) - (x*y0));
  double C = 2.0 * (x*y);
  double D = 2.0 * ((y*y0) - (y*x0));
  double E = y.lengthSquare();
  double F = x0.lengthSquare() + y0.lengthSquare();
  double denom = (C*C - 4*A*E);
  if (denom == 0.0) {
    // lines are parallel:
    double dist = distanceToLine(x0, y0, y);
    double lt = 0;// arbitray point on first line
    Vector3D v = x0 - y0;
    double d0 = v.length(); // distance between start points
    if (d0 == 0.0) {
      return 0.0; // lines are parallel and touch (meaning they are identical) and have some offset point
    }
    double alpha = angle(v, y);
    double ls = cos(alpha) * d0; // projection on second line
    return Vector3D(lt, ls, dist); // lines are probably parallel : no unique solution possible, choose t = 0
  }
  double s = ((2*A*D) + (B*C)) / denom;
  double t = (C*s - B) / (2.0*A); 
  double d = sqrt( (B*C*D+B*B*E+C*C*F+A*(D*D-4.0*E*F))/denom );
  return Vector3D(t, s, d);
}

/** return minimum distance between two line segements.
 * Line segment A defined with end points a0 and a1,
 * Line segment B defined with end points b0 and b1.
 */

Vector3D 
computeLineSegmentDistance(const Vector3D& a0,
			   const Vector3D& a1,
			   const Vector3D& b0, 
			   const Vector3D& b1) {
  Vector3D dirA = a1 - a0;
  Vector3D dirB = b1 - b0;
  double lenA = dirA.length();
  double lenB = dirB.length();
  ASSERT(dirA.lengthSquare() > 0.0);
  ASSERT(dirB.lengthSquare() > 0.0);
  dirA.normalize();
  dirB.normalize();
  Vector3D distVec = computeLineDistance(a0, dirA, b0, dirB);
  double t = distVec.x(); // point on first line = a0 + t * dirA
  double s = distVec.y(); // point on first line = a0 + t * dirA
  if (t < 0.0) {
    t = 0.0; // out of range of line segment
  }
  else if (t > lenA) {
    t = lenA;
  }
  if (s < 0.0) {
    s = 0.0; // out of range of line segment, set to extreme values
  }
  else if (s > lenB) {
    s = lenB;
  }
  Vector3D pA = a0 + (dirA * t); // actual minimum points in space
  Vector3D pB = b0 + (dirB * t);
  double d = vecDistance(pA, pB);
  return Vector3D(t, s, d);
}
