// --*- C++ -*------x---------------------------------------------------------
// $Id: 
//
// Class:           Random
// 
// Base class:      -
//
// Derived classes: - 
//
// Author:          Matthias Heiler, Eckart Bindewald
//
// Project name:    -
//
// Description:     Replaces stdlib's rand() function. The implementation as 
//                  Singelton provides the random number generator from 
//                  beeing re-seeded over and over again. 
// 
// -----------------x-------------------x-------------------x-----------------

#ifndef __RANDOM_H__
#define __RANDOM_H__

// Includes

extern "C" {
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
}

/** Random number generator.

    Replaces stdlib's rand() function. The implementation as Singleton
    provides the random number generator from beeing re-seeded over
    and over again.

    @see    Gamma et al: "Design Patterns", Singleton-Pattern */
class Random {
public:

  typedef unsigned int seed_type;

  /* OPERATORS */
  /** "functor", use for example with random_shuffle */
  long operator () (long n)  { return getRand(n); }

  /** Return reference to RNG object. */
  static Random& getInstance();

  /** How big is biggest number we can get? */
  static inline long getMax();

  /** What is next random number? */
  long getRand();

  /** return random number between 0 and n-1 : */
  long getRand(long n) { 
    if (n < 2) {
      return 0;
    }
    return getRand() % n; 
  }

  /** What is next random number?. 
      This method returns a float in [0..1[. */
  double getRandf();

  /** return normally distributed random number */
  double getGaussian();

  /** Get exponentially distributed random number using formula -gamma ln(x)
   * @see http://www.sitmo.com/eq/513
   */
  double getExponential();

  /* MODIFIERS */

  /**
   * resets random number generator and sets seed 
   */
  void resetWithSeed(seed_type seed);

protected:
  Random();

private:
  static Random* msInstance;
  int iset;
  double gset;
};

/** initialize random number generator using time and process id as seed */
inline 
Random::Random() : iset(0), gset(0.0)

{
  srand((int)getpid() + time(0));
}

inline 
void
Random::resetWithSeed(seed_type seed)
{
  iset = 0;
  gset = 0.0;
  srand(seed);
}

inline 
Random&
Random::getInstance()
{
  if (msInstance == 0)
    {
      msInstance = new Random;
    }

  return *msInstance;
}

inline
long
Random::getRand()
{
  return ::rand();
}

inline
double
Random::getRandf()
{
  return ::rand() / static_cast<double>(RAND_MAX);
}

inline
long
Random::getMax()
{
  return RAND_MAX;
}

#endif /* __RADNOM_H__ */

